/* ═══════════════════════════════════════════════════════════
   ZipIt — popup.js
   Lite  : HTML + CSS + JS, fast, images/fonts load from CDN
   Full  : Every asset downloaded + ALL paths rewritten so
           index.html opens 100% offline from the ZIP folder
   ═══════════════════════════════════════════════════════════ */

let currentMode = 'lite';
let currentUrl = '';
let currentTab = null;
let extractionCount = 0;
let isPro = false;

// ── Theme Management ──────────────────────────────────────────
async function toggleTheme() {
  const isLight = document.body.classList.toggle('light-mode');
  await chrome.storage.local.set({ theme: isLight ? 'light' : 'dark' });
  updateThemeIcons(isLight);
}

async function loadTheme() {
  const { theme } = await chrome.storage.local.get('theme');
  const isLight = theme === 'light';
  document.body.classList.toggle('light-mode', isLight);
  updateThemeIcons(isLight);
}

function updateThemeIcons(isLight) {
  document.getElementById('theme-icon-sun').style.display = isLight ? 'block' : 'none';
  document.getElementById('theme-icon-moon').style.display = isLight ? 'none' : 'block';
}

// ── Mode Switching ──────────────────────────────────────────
function switchMode(mode) {
  currentMode = mode;

  // Tabs
  document.getElementById('tab-lite').classList.toggle('active', mode === 'lite');
  document.getElementById('tab-full').classList.toggle('active', mode === 'full');
  document.getElementById('tab-design').classList.toggle('active', mode === 'design');

  // Panels
  document.getElementById('panel-lite').classList.toggle('active', mode === 'lite');
  document.getElementById('panel-full').classList.toggle('active', mode === 'full');
  document.getElementById('panel-analysis').classList.toggle('active', mode === 'design');

  // Logic visibility
  const actionWrap = document.querySelector('.actions');
  const progressWrap = document.getElementById('progress-wrap');

  if (mode === 'design') {
    actionWrap.style.display = 'none';
    progressWrap.classList.remove('visible');
    analyzeDesignSystem(); // Auto-analyze when clicking the tab
  } else {
    actionWrap.style.display = 'flex';
    document.getElementById('btn-lite').style.display = (mode === 'lite') ? 'flex' : 'none';
    document.getElementById('btn-full').style.display = (mode === 'full') ? 'flex' : 'none';
  }
}
async function refreshContext(manual = false) {
  if (manual) {
    const btn = document.getElementById('btn-refresh');
    btn.style.transition = 'transform 0.5s ease';
    btn.style.transform = 'rotate(360deg)';
    setTimeout(() => btn.style.transform = 'rotate(0deg)', 510);
  }

  chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
    if (!tabs?.[0]) return;
    currentTab = tabs[0];
    currentUrl = tabs[0].url;

    const displayUrl = currentUrl.replace(/^https?:\/\//, '').split('?')[0].split('#')[0];
    document.getElementById('current-url').textContent = displayUrl;

    if (currentUrl.startsWith('http')) {
      document.getElementById('url-dot').classList.add('on');
      document.getElementById('btn-lite').disabled = false;
      document.getElementById('btn-full').disabled = false;

      // If we are already in analysis mode, auto-re-analyze the new site
      if (currentMode === 'design') {
        analyzeDesignSystem();
      }
    } else {
      document.getElementById('url-dot').classList.remove('on');
      document.getElementById('btn-lite').disabled = true;
      document.getElementById('btn-full').disabled = true;
    }
  });
}

// ── Init ─────────────────────────────────────────────────────
document.addEventListener('DOMContentLoaded', async () => {
  // Load stats and Pro status
  const data = await chrome.storage.local.get(['extractionCount', 'isPro']);
  extractionCount = data.extractionCount || 0;
  isPro = data.isPro || false;

  updateStatsUI();
  checkPaywallStatus();
  refreshContext();

  // Automatic refresh when switching tabs
  chrome.tabs.onActivated.addListener(() => refreshContext());
  chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
    if (changeInfo.status === 'complete') refreshContext();
  });

  // Tab listeners
  document.getElementById('tab-lite').addEventListener('click', () => switchMode('lite'));
  document.getElementById('tab-full').addEventListener('click', () => switchMode('full'));
  document.getElementById('tab-design').addEventListener('click', () => switchMode('design'));

  // Dashboard Tab switching
  document.querySelectorAll('.dash-tab').forEach(tab => {
    tab.addEventListener('click', () => {
      const target = tab.getAttribute('data-tab');
      document.querySelectorAll('.dash-tab').forEach(t => t.classList.remove('active'));
      tab.classList.add('active');

      document.querySelectorAll('.dash-section').forEach(sec => sec.style.display = 'none');
      document.getElementById(`section-${target}`).style.display = 'block';
    });
  });

  // Fonts Sub-tab switching
  document.querySelectorAll('.font-sub-tab').forEach(tab => {
    tab.addEventListener('click', () => {
      const target = tab.getAttribute('data-sub');
      document.querySelectorAll('.font-sub-tab').forEach(t => t.classList.remove('active'));
      tab.classList.add('active');

      document.getElementById('sub-section-families').style.display = target === 'families' ? 'block' : 'none';
      document.getElementById('sub-section-scale').style.display = target === 'scale' ? 'block' : 'none';
    });
  });

  // Export buttons
  document.getElementById('btn-copy-figma').addEventListener('click', () => copyFigmaTokens());
  document.getElementById('btn-copy-prompt').addEventListener('click', () => copyAIPrompt());
  document.getElementById('btn-export-tailwind').addEventListener('click', () => exportDesign('tailwind'));
  document.getElementById('btn-export-css-full').addEventListener('click', () => exportDesign('css'));
  document.getElementById('btn-export-json').addEventListener('click', () => exportDesign('json'));

  document.getElementById('btn-copy-palette').addEventListener('click', () => {
    if (currentAnalysis) {
      const allHex = currentAnalysis.colors.join('\n');
      copyToClipboard(allHex, 'All colors copied to clipboard!');
    }
  });

  // SidePanel Trigger
  document.getElementById('btn-sidepanel').addEventListener('click', () => {
    chrome.sidePanel.open({ tabId: currentTab.id });
    window.close();
  });


  // Theme Toggle
  document.getElementById('btn-theme').addEventListener('click', () => toggleTheme());
  loadTheme();

  // Refresh Button
  document.getElementById('btn-refresh').addEventListener('click', () => refreshContext(true));

  document.getElementById('btn-open-playground').addEventListener('click', () => {
    chrome.tabs.create({ url: chrome.runtime.getURL('playground.html') });
  });
  document.getElementById('btn-copy-component').addEventListener('click', () => captureFigmaSystemSync());

  document.getElementById('btn-lite').addEventListener('click', () => runDownload('lite'));
  document.getElementById('btn-full').addEventListener('click', () => runDownload('full'));

  // Paywall / Pro Listeners
  document.getElementById('btn-upgrade-lifetime').addEventListener('click', () => {
    // PayPal/Checkout Integration
    const payPalCheckoutUrl = 'https://www.paypal.com/ncp/payment/ARMVRY4T34UKY';
    chrome.tabs.create({ url: payPalCheckoutUrl });
  });

  // Switch from Paywall to License Entry
  document.getElementById('btn-show-license').addEventListener('click', () => {
    document.getElementById('paywall-overlay').classList.remove('active');
    document.getElementById('license-overlay').classList.add('active');
    document.getElementById('license-key-input').focus();
  });

  // Cancel License Entry
  document.getElementById('btn-cancel-license').addEventListener('click', () => {
    document.getElementById('license-overlay').classList.remove('active');
    document.getElementById('paywall-overlay').classList.add('active');
  });

  // Submit and Validate License
  document.getElementById('btn-submit-license').addEventListener('click', () => {
    const input = document.getElementById('license-key-input');
    const error = document.getElementById('license-error');
    const key = input.value.trim().toUpperCase();

    // 10 TEST KEYS FOR ZIPIT V1.0
    const validKeys = [
      'ZIPIT-PRO-2026',
      'ZIPIT-ULTRA-OFFLINE',
      'ZIPIT-TEST-DEV-01',
      'ZIPIT-PREMIUM-SCAN',
      'ZIPIT-LIFETIME-PRO',
      'ZIPIT-DESIGN-INTEL',
      'ZIPIT-BETA-RELEASE',
      'ZIPIT-UNLIMITED-X',
      'ZIPIT-PRO-MAX-PLUS',
      'ZIPIT-GIBBERISH-FIX'
    ];

    if (validKeys.includes(key)) {
      unlockPro();
      document.getElementById('license-overlay').classList.remove('active');
      showStatus('Pro Mode Activated! Enjoy unlimited downloads.', 'success');
    } else {
      error.style.display = 'block';
      input.style.borderColor = 'var(--red)';
      setTimeout(() => {
        input.style.borderColor = 'var(--border)';
      }, 1000);
    }
  });

  // Clear error on type
  document.getElementById('license-key-input').addEventListener('input', () => {
    document.getElementById('license-error').style.display = 'none';
    document.getElementById('license-key-input').style.borderColor = 'var(--border)';
  });

  console.log('ZipIt UI Loaded - Paywall & Pro Active');
});

function updateStatsUI() {
  const footerLeft = document.getElementById('footer-left');
  if (!isPro) {
    footerLeft.textContent = `${extractionCount}/10 Free Extractions`;
  } else {
    footerLeft.textContent = `Pro Lifetime Active`;
  }
}

function checkPaywallStatus() {
  if (!isPro && extractionCount >= 10) {
    document.getElementById('paywall-overlay').classList.add('active');
  }
}

async function incrementExtraction() {
  if (isPro) return;
  extractionCount++;
  await chrome.storage.local.set({ extractionCount });
  updateStatsUI();
  checkPaywallStatus();
}

async function unlockPro() {
  isPro = true;
  await chrome.storage.local.set({ isPro: true });
  updateStatsUI();
  document.getElementById('paywall-overlay').classList.remove('active');
  showStatus('✨ Welcome to Pro! All limits removed.', 'success');
  launchConfetti();
}

// ── Analysis & Export (Universal Design Intelligence) ────────
let currentAnalysis = null;

async function analyzeDesignSystem() {
  if (!currentTab) return;
  hideStatus();

  // Site info
  document.getElementById('analysis-site-name').textContent = 'Analyzing Page...';
  document.getElementById('analysis-site-url').textContent = currentUrl;

  try {
    const [{ result: analysis }] = await chrome.scripting.executeScript({
      target: { tabId: currentTab.id },
      func: performDeepAnalysis
    });

    currentAnalysis = analysis;
    renderDashboard(analysis);
    showStatus(`✓ Analysis complete: ${analysis.colors.length} colors, ${analysis.media.length} visual assets`, 'success');
  } catch (err) {
    document.querySelectorAll('.stat-value').forEach(s => s.textContent = '0');
    showStatus(`Analysis failed: ${err.message}`, 'error');
  }
}





async function captureFigmaSystemSync() {
  if (!currentTab) return;
  window.close(); // Close extension to pick

  await chrome.scripting.executeScript({
    target: { tabId: currentTab.id },
    func: () => {
      // 1. Setup Selection UI
      const style = document.createElement('style');
      style.id = 'zipit-system-sync';
      style.textContent = `
        .zipit-sync-h { outline: 3px solid #6c5ce7 !important; outline-offset: -3px !important; cursor: crosshair !important; background: rgba(108, 92, 231, 0.1) !important; transition: all 0.1s ease; }
        #zipit-sync-bar { position: fixed; top: 20px; left: 50%; transform: translateX(-50%); background: #1e1e1e; color: #fff; padding: 14px 28px; border-radius: 50px; z-index: 2147483647; font-family: Inter, sans-serif; font-size: 13px; font-weight: 600; box-shadow: 0 15px 45px rgba(0,0,0,0.6); display: flex; align-items: center; gap: 10px; border: 1px solid rgba(255,255,255,0.1); pointer-events: none; }
      `;
      document.head.appendChild(style);

      const syncBar = document.createElement('div');
      syncBar.id = 'zipit-sync-bar';
      syncBar.innerHTML = '⚙️ <b>System Sync:</b> Pick Component for Auto-Layout Extraction';
      document.body.appendChild(syncBar);

      const onHover = (e) => {
        e.stopPropagation();
        document.querySelectorAll('.zipit-sync-h').forEach(x => x.classList.remove('zipit-sync-h'));
        e.target.classList.add('zipit-sync-h');
      };

      const onClick = async (e) => {
        e.preventDefault(); e.stopPropagation();
        const rootEl = e.target;

        // Convert any relative image URLs to absolute so they work when pasted
        const clone = rootEl.cloneNode(true);
        const images = clone.querySelectorAll('img');
        const sourceImages = rootEl.querySelectorAll('img');
        images.forEach((img, i) => {
          if (sourceImages[i].src) {
            img.src = new URL(sourceImages[i].src, document.baseURI).href;
          }
        });



        // Recursively apply computed styles so the component renders exactly as it did on the source page
        const inlinify = (source, target) => {
          if (source.nodeType !== 1) return;
          const s = window.getComputedStyle(source);
          let css = '';
          const props = [
            'box-sizing', 'display', 'flex-direction', 'justify-content', 'align-items', 'gap', 'flex-wrap',
            'padding-top', 'padding-right', 'padding-bottom', 'padding-left',
            'margin-top', 'margin-right', 'margin-bottom', 'margin-left',
            'width', 'height', 'max-width', 'max-height', 'min-width', 'min-height',
            'background-color', 'background-image', 'background-size', 'background-position', 'background-repeat',
            'color', 'font-size', 'font-weight', 'font-family', 'line-height', 'text-align', 'letter-spacing',
            'border-top-width', 'border-right-width', 'border-bottom-width', 'border-left-width',
            'border-top-style', 'border-right-style', 'border-bottom-style', 'border-left-style',
            'border-top-color', 'border-right-color', 'border-bottom-color', 'border-left-color',
            'border-top-left-radius', 'border-top-right-radius', 'border-bottom-right-radius', 'border-bottom-left-radius',
            'box-shadow', 'opacity', 'position', 'top', 'right', 'bottom', 'left', 'z-index',
            'transform', 'overflow'
          ];

          props.forEach(p => {
            let val = s.getPropertyValue(p);
            if (val && val !== 'none' && val !== 'normal' && val !== 'auto' && val !== '0px' && val !== 'rgba(0, 0, 0, 0)' && val !== 'matrix(1, 0, 0, 1, 0, 0)' && val !== 'all 0s ease 0s' && val !== 'auto') {
              if (p === 'font-family') val = val.replace(/"/g, "'");
              css += p + ': ' + val + '; ';
            }
          });

          if (css) {
            target.setAttribute('style', target.getAttribute('style') ? target.getAttribute('style') + '; ' + css : css);
          }

          // Remove messy external classes
          target.removeAttribute('class');
          target.removeAttribute('id');

          for (let i = 0; i < source.children.length; i++) {
            inlinify(source.children[i], target.children[i]);
          }
        };

        inlinify(rootEl, clone);

        // Get the HTML and format it like the Stitch AI output
        const componentHTML = clone.outerHTML;

        const payload = `<!DOCTYPE html>
<html class="light" lang="en">
<head>
    <meta charset="utf-8"/>
    <meta content="width=device-width, initial-scale=1.0" name="viewport"/>
    <title>ZipIt Extracted Component</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet"/>
    <link href="https://fonts.googleapis.com/css2?family=Material+Symbols+Outlined:wght,FILL@100..700,0..1&display=swap" rel="stylesheet"/>
    <script src="https://cdn.tailwindcss.com?plugins=forms,typography"></script>
    <style type="text/tailwindcss">
        body { font-family: 'Inter', sans-serif; }
    </style>
</head>
<body class="bg-white text-slate-900">
    <!-- Extracted Component -->
    ${componentHTML}
</body>
</html>`;

        await navigator.clipboard.writeText(payload);
        const bar = document.getElementById('zipit-sync-bar');
        bar.style.background = '#22c55e';
        bar.innerHTML = '✨ <b>HTML Code Copied!</b> You can now paste it into your tool.';
        setTimeout(() => {
          document.removeEventListener('mouseover', onHover, true);
          document.removeEventListener('click', onClick, true);
          document.getElementById('zipit-system-sync')?.remove();
          document.getElementById('zipit-sync-bar')?.remove();
          document.querySelectorAll('.zipit-sync-h').forEach(x => x.classList.remove('zipit-sync-h'));
        }, 3000);
      };
      document.addEventListener('mouseover', onHover, true);
      document.addEventListener('click', onClick, true);
    }
  });

  launchConfetti();
}

async function copyFigmaLayers() {
  if (!currentTab) return;
  showStatus('📸 Flattening whole page for Figma...', 'info');

  await chrome.scripting.executeScript({
    target: { tabId: currentTab.id },
    func: () => {
      let svgContent = "";
      const crawl = (el) => {
        if (el.nodeType !== 1) return "";
        const s = window.getComputedStyle(el);
        if (s.display === 'none' || s.visibility === 'hidden' || s.opacity === '0') return "";

        const r = el.getBoundingClientRect();
        if (r.width < 1 || r.height < 1) return "";

        const lyr = (el.className.split(' ')[0] || el.tagName).replace(/[^a-zA-Z0-9]/g, '_');
        let inner = "";

        if (s.backgroundColor !== 'rgba(0, 0, 0, 0)' && s.backgroundColor !== 'transparent') {
          inner += `<rect id="${lyr}_bg" x="${r.left}" y="${r.top}" width="${r.width}" height="${r.height}" fill="${s.backgroundColor}" rx="${parseInt(s.borderRadius) || 0}" />\n`;
        }
        if (parseInt(s.borderWidth) > 0) {
          inner += `<rect id="${lyr}_border" x="${r.left}" y="${r.top}" width="${r.width}" height="${r.height}" fill="none" stroke="${s.borderColor}" stroke-width="${s.borderWidth}" rx="${parseInt(s.borderRadius) || 0}" />\n`;
        }
        if (el.tagName === 'IMG' && el.src) {
          inner += `<image id="${lyr}_img" x="${r.left}" y="${r.top}" width="${r.width}" height="${r.height}" href="${new URL(el.src, document.baseURI).href}" />\n`;
        }
        if (el.children.length === 0 && el.textContent.trim()) {
          const txt = el.textContent.trim().replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;');
          inner += `<text id="${lyr}_txt" x="${r.left}" y="${r.top + (r.height * 0.8)}" font-family="${s.fontFamily.replace(/"/g, "'")}" font-size="${s.fontSize}" fill="${s.color}">${txt}</text>\n`;
        }

        let childSvg = "";
        for (let i = 0; i < el.children.length; i++) { childSvg += crawl(el.children[i]); }
        return `<g id="${lyr}">${inner}${childSvg}</g>\n`;
      };

      const finalSvg = `<svg xmlns="http://www.w3.org/2000/svg" width="${window.innerWidth}" height="${window.innerHeight}"><rect width="100%" height="100%" fill="white" />${crawl(document.body)}</svg>`;
      const blob = new Blob([finalSvg], { type: 'image/svg+xml;charset=utf-8' });
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url; a.download = `zipit_viewport_${new Date().getTime()}.svg`;
      a.click();
      setTimeout(() => URL.revokeObjectURL(url), 10000);
    }
  });

  showStatus('🖼️ Figma Viewport Ready!', 'success');
  launchConfetti();
}
function renderDashboard(data) {
  document.getElementById('analysis-site-name').textContent = data.title;
  document.getElementById('dash-stat-colors').textContent = data.colors.length;
  document.getElementById('dash-stat-fonts').textContent = data.fonts.length;
  document.getElementById('dash-stat-media').textContent = data.media.filter(m => m.type === 'image').length;
  document.getElementById('dash-stat-svg').textContent = data.media.filter(m => m.type === 'svg').length;

  // Overview Palette
  const palette = document.getElementById('overview-palette');
  palette.innerHTML = '';
  data.colors.slice(0, 10).forEach(hex => {
    const div = document.createElement('div');
    div.style.flex = '1';
    div.style.backgroundColor = hex;
    palette.appendChild(div);
  });

  // Colors Grid
  const gridColors = document.getElementById('grid-colors');
  gridColors.innerHTML = '';
  data.colors.forEach((hex, i) => {
    const card = document.createElement('div');
    card.className = 'color-card';
    const contrast = getContrast(hex, '#000000') > getContrast(hex, '#ffffff') ? 'black' : 'white';
    const ratio = getContrast(hex, contrast === 'white' ? '#ffffff' : '#000000');
    const badge = ratio > 4.5 ? '<span class="contrast-badge contrast-pass">AAA Pass</span>' : '<span class="contrast-badge contrast-fail">Low Contrast</span>';

    card.innerHTML = `
      <div class="color-preview" style="background:${hex};"></div>
      <div class="color-info">
        <div style="display:flex; align-items:center;">
          <div class="color-name">color-brand-${i + 1}</div>
          ${badge}
        </div>
        <div class="color-hex">${hex}</div>
      </div>
    `;
    card.onclick = () => {
      copyToClipboard(hex, `Hex ${hex} copied!`);
      card.classList.add('copied');
      setTimeout(() => card.classList.remove('copied'), 1500);
    };
    gridColors.appendChild(card);
  });

  // Fonts List
  const listFonts = document.getElementById('list-fonts');
  listFonts.innerHTML = '';



  // 2. Font Families Section
  const familyTitle = document.createElement('div');
  familyTitle.className = 'section-title';
  familyTitle.style.marginTop = '24px';
  familyTitle.textContent = 'Detected Families';
  listFonts.appendChild(familyTitle);

  data.fonts.forEach(f => {
    const alternative = getGoogleAlternative(f);
    const item = document.createElement('div');
    item.className = 'font-item';
    item.innerHTML = `
      <div class="font-aa" style="font-family: ${f};">Aa</div>
      <div class="font-info">
        <div class="font-family">${f}</div>
        <div class="font-meta">Regular, Medium, Bold</div>
        <div class="font-suggest">
          <span>Similar: </span>
          <a href="https://fonts.google.com/specimen/${alternative.replace(/ /g, '+')}" target="_blank">${alternative}</a>
        </div>
      </div>
    `;
    listFonts.appendChild(item);
  });

  // 3. Type Scale Section
  const listTypeScale = document.getElementById('list-typescale');
  listTypeScale.innerHTML = '';

  const sortedTypo = data.typography.sort((a, b) => b.size - a.size);
  const ratios = [];

  for (let i = 0; i < sortedTypo.length - 1; i++) {
    const r = sortedTypo[i].size / sortedTypo[i + 1].size;
    if (r > 1 && r < 2) ratios.push(r);
  }

  const medianRatio = ratios.length > 0 ? (ratios.reduce((a, b) => a + b, 0) / ratios.length).toFixed(3) : "1.250";
  const baseSize = data.typography.find(t => t.tag === 'p')?.size || 16;

  document.getElementById('typescale-factor').textContent = medianRatio;
  document.getElementById('typescale-base').textContent = baseSize + 'px';

  sortedTypo.forEach(t => {
    const item = document.createElement('div');
    item.className = 'typo-item';
    item.innerHTML = `
      <div class="typo-tag">${t.tag}</div>
      <div class="typo-preview" style="font-family: ${t.family}; font-size: ${t.size}px; font-weight: ${t.weight}; line-height: ${t.lh};">
        Typography Hierarchy
      </div>
      <div class="typo-meta">${t.size}px / ${t.lh} — ${t.family.split(',')[0]}</div>
    `;
    listTypeScale.appendChild(item);
  });

  // Media Grid
  const gridMedia = document.getElementById('grid-media');
  gridMedia.innerHTML = '';
  data.media.forEach(m => {
    const card = document.createElement('div');
    card.className = 'media-card';
    card.innerHTML = `
      <div class="media-thumb" style="background-image: url('${m.url}');" draggable="true"></div>
      <div class="media-dl" title="Download Asset">
        <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round">
          <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
          <polyline points="7 10 12 15 17 10"></polyline>
          <line x1="12" y1="15" x2="12" y2="3"></line>
        </svg>
      </div>
    `;

    const thumb = card.querySelector('.media-thumb');
    thumb.addEventListener('dragstart', (e) => {
      e.dataTransfer.setData('text/plain', m.url);
      e.dataTransfer.setDragImage(thumb, 50, 50);
    });

    card.querySelector('.media-dl').onclick = (e) => {
      e.stopPropagation();
      downloadSingleAsset(m.url);
    };
    gridMedia.appendChild(card);
  });
}

function copyToClipboard(text, msg) {
  navigator.clipboard.writeText(text);
  showStatus(msg, 'success');
}

function downloadSingleAsset(url) {
  const filename = url.split('/').pop().split('?')[0] || 'asset';
  chrome.downloads.download({ url, filename });
}

function exportDesign(format) {
  if (!currentAnalysis) return;
  const data = currentAnalysis;
  let content = '';
  let filename = `design_system.${format === 'json' ? 'json' : format === 'css' ? 'css' : 'js'}`;

  if (format === 'json') {
    content = JSON.stringify({
      site: data.title,
      colors: data.colors,
      fonts: data.fonts,
      mediaCount: data.media.length
    }, null, 2);
  } else if (format === 'css') {
    content = `:root {\n` + data.colors.map((c, i) => `  --color-brand-${i + 1}: ${c};`).join('\n') + `\n}\n\n`;
    content += `body {\n  font-family: "${data.fonts[0] || 'Inter'}", sans-serif;\n}`;
  } else if (format === 'tailwind') {
    content = `module.exports = {\n  theme: {\n    extend: {\n      colors: {\n` +
      data.colors.map((c, i) => `        'brand-${i + 1}': '${c}',`).join('\n') +
      `\n      },\n      fontFamily: {\n        'brand': ['${data.fonts[0] || 'Inter'}'],\n      }\n    }\n  }\n}`;
  }

  const blob = new Blob([content], { type: 'text/plain' });
  const fileUrl = URL.createObjectURL(blob);
  chrome.downloads.download({ url: fileUrl, filename, saveAs: true });
}

function copyAIPrompt() {
  if (!currentAnalysis) return;
  const data = currentAnalysis;

  const prompt = `Act as an expert UI Developer. I want to build a new section for my website project using the design system analyzed from ${data.title}.

PROJECT CONTEXT:
- Site: ${data.title}
- Source: ${currentUrl}

DESIGN SYSTEM TOKENS:
1. Colors: ${data.colors.join(', ')}
2. Typography: The primary font is "${data.fonts[0] || 'Inter'}", with fallbacks: ${data.fonts.slice(1, 5).join(', ')}.

TASK:
Create a modern, high-conversion [SECTION NAME] component using these specific colors and fonts. Ensure the design feels premium, use subtle gradients using the palette, and ensure accessibility. Provide the clean HTML and Tailwind CSS code.`;

  navigator.clipboard.writeText(prompt);
  showStatus('✨ AI Prompt copied to clipboard!', 'success');
}

function copyFigmaTokens() {
  if (!currentAnalysis) return;
  const data = currentAnalysis;

  const tokens = {
    "Color": {},
    "Font Family": {}
  };

  data.colors.forEach((hex, i) => {
    tokens["Color"][`brand-${i + 1}`] = {
      "value": hex,
      "type": "color"
    };
  });

  if (data.fonts.length > 0) {
    tokens["Font Family"]["primary"] = {
      "value": data.fonts[0],
      "type": "fontFamilies"
    };
  }

  const json = JSON.stringify(tokens, null, 2);
  navigator.clipboard.writeText(json);
  showStatus('🎨 Figma Design Tokens copied!', 'success');
}

function getContrast(hex1, hex2) {
  const l1 = getLuminance(hex1);
  const l2 = getLuminance(hex2);
  return (Math.max(l1, l2) + 0.05) / (Math.min(l1, l2) + 0.05);
}

function getLuminance(hex) {
  const rgb = hex.match(/[A-Za-z0-9]{2}/g).map(x => parseInt(x, 16) / 255);
  const [r, g, b] = rgb.map(v => v <= 0.03928 ? v / 12.92 : Math.pow((v + 0.055) / 1.055, 2.4));
  return 0.2126 * r + 0.7152 * g + 0.0722 * b;
}

function getGoogleAlternative(font) {
  const f = font.toLowerCase();

  // Sans-Serif Mappings
  if (f.includes('arial') || f.includes('helvetica') || f.includes('sans-serif')) return 'Inter';
  if (f.includes('futura')) return 'Montserrat';
  if (f.includes('gill sans')) return 'Lato';
  if (f.includes('franklin gothic')) return 'Libre Franklin';
  if (f.includes('din') || f.includes('bahnschrift')) return 'Barlow';
  if (f.includes('segoe ui') || f.includes('roboto')) return 'Inter';
  if (f.includes('avoids') || f.includes('circular')) return 'Plus Jakarta Sans';

  // Serif Mappings
  if (f.includes('times') || f.includes('serif')) return 'PT Serif';
  if (f.includes('georgia')) return 'Playfair Display';
  if (f.includes('garamond')) return 'EB Garamond';
  if (f.includes('baskerville')) return 'Libre Baskerville';
  if (f.includes('caslon')) return 'Big Caslon';

  // Monospace
  if (f.includes('mono') || f.includes('courier') || f.includes('consolas')) return 'Roboto Mono';

  // Quality Fallback
  return 'Plus Jakarta Sans';
}

/** Injected Script: Performs a much deeper analysis */
function performDeepAnalysis() {
  const colors = new Set();
  const fonts = new Set();
  const typography = [];
  const media = [];

  // Site Info
  const title = document.title;

  // 1. Scan Typography Scale
  const tags = ['h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'p', 'span', 'button', 'small', 'caption'];
  tags.forEach(tag => {
    const el = document.querySelector(tag);
    if (el) {
      const s = getComputedStyle(el);
      typography.push({
        tag: tag,
        size: parseInt(s.fontSize),
        weight: s.fontWeight,
        lh: s.lineHeight,
        family: s.fontFamily
      });
    }
  });

  const getStyle = (el, prop) => window.getComputedStyle(el).getPropertyValue(prop);

  const rgbToHex = (rgb) => {
    const m = rgb.match(/^rgba?\((\d+),\s*(\d+),\s*(\d+)(?:,\s*([\d.]+))?\)$/);
    if (!m || (m[4] && parseFloat(m[4]) === 0)) return null;
    const hex = (x) => ("0" + parseInt(x).toString(16)).slice(-2);
    return ("#" + hex(m[1]) + hex(m[2]) + hex(m[3])).toUpperCase();
  };

  // 1. COLORS: Scan everything
  document.querySelectorAll('*').forEach(el => {
    ['background-color', 'color', 'border-color', 'fill', 'stroke'].forEach(prop => {
      const val = getStyle(el, prop);
      const hex = rgbToHex(val);
      if (hex && hex !== '#00000000') colors.add(hex);
    });

    // Fonts
    const ff = getStyle(el, 'font-family').split(',')[0].replace(/['"]/g, '').trim();
    if (ff && ff !== 'inherit') fonts.add(ff);
  });

  // 2. MEDIA: Images and SVGs
  document.querySelectorAll('img, [style*="background-image"], svg').forEach(el => {
    let url = '';
    let type = 'image';

    if (el.tagName === 'IMG' && el.src) {
      url = el.src;
    } else if (el.tagName === 'svg') {
      type = 'svg';
    } else {
      const bg = getStyle(el, 'background-image');
      if (bg && bg !== 'none') {
        const m = bg.match(/url\(['"]?(.*?)['"]?\)/);
        if (m) url = m[1];
      }
    }

    if (url && (url.startsWith('http') || url.startsWith('data:image'))) {
      if (!media.find(m => m.url === url)) media.push({ url, type });
    }
  });

  return {
    title: document.title,
    colors: Array.from(colors).slice(0, 100).filter(c => c.startsWith('#')),
    fonts: Array.from(fonts).slice(0, 50),
    typography: typography,
    media: media.slice(0, 500)
  };
}


// ── Main download flow ────────────────────────────────────────
async function runDownload(mode) {
  if (!currentUrl) return;

  usedPaths.clear();
  setLoading(mode, true);
  hideStatus();
  showProgress('Scanning page assets…');
  updateProgress(5);

  try {
    // ── Step 1: extract asset URLs from live DOM ──────────────
    const selectedTypes = mode === 'lite' ? ['css', 'js'] : getSelectedTypes();

    const results = await chrome.scripting.executeScript({
      target: { tabId: currentTab.id },
      func: extractAssets,
      args: [selectedTypes],
    });

    const assets = (results && results[0] && results[0].result) ? results[0].result : [];

    updateProgress(12);
    showProgress(`Found ${assets.length} asset(s). Downloading…`);

    // Get the most up-to-date title from the live page
    let liveTitle = '';
    try {
      const [{ result }] = await chrome.scripting.executeScript({
        target: { tabId: currentTab.id },
        func: () => document.title,
      });
      liveTitle = result;
    } catch { }

    // Build a meaningful site name
    let siteName = liveTitle && liveTitle.trim() ? liveTitle.trim() : "";
    if (!siteName) {
      try { siteName = new URL(currentUrl).hostname.replace("www.", ""); }
      catch { siteName = "website"; }
    }
    const safeSiteName = siteName.replace(/[<>:"/\\|?*\x00-\x1f]/g, '_').replace(/\s+/g, '_').trim() || 'website';

    const rootFolder = safeSiteName.toLowerCase();
    const zipContainer = new JSZip();
    const zip = zipContainer.folder(rootFolder);
    const urlToLocal = new Map(); // original URL → local zip path

    // ── Step 2: fetch all assets ──────────────────────────────
    let fetched = 0, skipped = 0, totalBytes = 0;
    const CONCURRENCY = 5;

    async function downloadAsset(asset) {
      try {
        const res = await fetchWithTimeout(asset.url, 10000);
        if (!res.ok) throw new Error(`HTTP ${res.status}`);
        const blob = await res.blob();
        totalBytes += blob.size;

        const localPath = makeLocalPath(asset.url, asset.type);
        urlToLocal.set(asset.url, localPath);
        zip.file(localPath, blob);
        fetched++;
      } catch (e) {
        skipped++;
      } finally {
        const done = fetched + skipped;
        updateProgress(Math.round(12 + (done / Math.max(assets.length, 1)) * 60));
        document.getElementById('progress-files').textContent =
          `${fetched} downloaded · ${skipped} skipped · ${formatBytes(totalBytes)}`;
      }
    }

    // Process in batches
    for (let i = 0; i < assets.length; i += CONCURRENCY) {
      const batch = assets.slice(i, i + CONCURRENCY);
      await Promise.all(batch.map(downloadAsset));
    }

    // ── Step 3: Capture + process HTML ─────────────────────────
    showProgress('Processing HTML…');
    updateProgress(74);

    // Get live DOM instead of fetching raw source (Better SPA support)
    const [{ result: liveHtml }] = await chrome.scripting.executeScript({
      target: { tabId: currentTab.id },
      func: () => document.documentElement.outerHTML,
    });

    let html = '<!DOCTYPE html>\n' + liveHtml;

    if (mode === 'full') {
      // Rewrite CSS and JS files: fix absolute URLs inside them
      showProgress('Rewriting asset paths…');
      updateProgress(78);

      const cssAssets = [...urlToLocal.entries()].filter(([url, path]) => path.startsWith('css/'));
      const jsAssets = [...urlToLocal.entries()].filter(([url, path]) => path.startsWith('js/'));

      // Process CSS
      const usedClasses = new Set();
      html.match(/class=(['"])(.*?)\1/gi)?.forEach(m => {
        const c = m.split('=')[1].replace(/['"]/g, '').split(/\s+/);
        c.forEach(cls => usedClasses.add(cls));
      });

      await Promise.all(cssAssets.map(async ([origUrl, localPath]) => {
        try {
          const cssRes = await fetchWithTimeout(origUrl, 8100);
          let cssText = await cssRes.text();
          cssText = rewriteCss(cssText, origUrl, urlToLocal);

          // Smart Purge if Pro
          if (isPro) {
            cssText = cssText.replace(/\.([a-z0-9_-]+)\s*\{[^}]*\}/gi, (match, cls) => {
              if (usedClasses.has(cls) || cls.startsWith('hover:') || cls.startsWith('focus:')) return match;
              return `/* Purged: .${cls} */`;
            });
          }

          zip.file(localPath, cssText);
        } catch { }
      }));

      // Process JS (Deep Chunk Rewrite)
      showProgress('Rewriting JS modules…');
      await Promise.all(jsAssets.map(async ([origUrl, localPath]) => {
        try {
          const jsRes = await fetchWithTimeout(origUrl, 8200);
          let jsContent = await jsRes.text();
          for (const [abs, local] of urlToLocal) {
            if (jsContent.includes(abs)) {
              const rel = local.startsWith('js/') ? local.slice(3) : '../' + local;
              jsContent = jsContent.split(abs).join(rel);
            }
          }
          zip.file(localPath, jsContent);
        } catch { }
      }));

      // Full rewrite of HTML
      html = rewriteHtmlFull(html, currentUrl, urlToLocal);
    } else {
      // Lite: only rewrite <link href> and <script src>
      html = rewriteHtmlLite(html, currentUrl, urlToLocal);
    }

    zip.file('index.html', html);

    // ── Apply UI UI Pro Max files if selected ─────────────────
    if (mode === 'full') {
      const styleInputEl = document.getElementById('import-stylesheet');
      const designInputEl = document.getElementById('import-design-system');

      if (styleInputEl && styleInputEl.files.length > 0) {
        const text = await styleInputEl.files[0].text();
        zip.file('ui-ux-pro-max/stylesheet.md', text);
      }
      if (designInputEl && designInputEl.files.length > 0) {
        const text = await designInputEl.files[0].text();
        zip.file('ui-ux-pro-max/design_system.md', text);
      }
    }

    // ── Step 4: generate ZIP ──────────────────────────────────
    showProgress('Compressing ZIP…');
    updateProgress(88);

    const zipBlob = await zipContainer.generateAsync(
      { type: 'blob', compression: 'DEFLATE', compressionOptions: { level: 6 } },
      (meta) => {
        document.getElementById('progress-files').textContent =
          `Compressing… ${Math.round(meta.percent)}%`;
      }
    );

    updateProgress(100);

    // ── Step 5: trigger download ──────────────────────────────
    const blobUrl = URL.createObjectURL(zipBlob);
    const filename = `${safeSiteName}.zip`;

    chrome.downloads.download({ url: blobUrl, filename, saveAs: false }, () => {
      setTimeout(() => URL.revokeObjectURL(blobUrl), 60000);
      incrementExtraction();
    });

    const total = fetched + 1; // +1 for index.html
    document.getElementById('footer-left').textContent = `${total} files · ${mode}`;

    hideProgress();
    showStatus(
      `✓ ${total} files · ${formatBytes(zipBlob.size)}${skipped ? ` · ${skipped} skipped` : ''}`,
      'success'
    );
    setLoading(mode, false);

  } catch (err) {
    hideProgress();
    setLoading(mode, false);
    showStatus(`Error: ${err.message}`, 'error');
  }
}

// ── HTML rewriting ────────────────────────────────────────────

/** Full mode — replace every src/href/url() with local paths */
function rewriteHtmlFull(html, baseUrl, urlToLocal) {
  const base = new URL(baseUrl);

  // src / href / posters / data-attributes
  html = html.replace(
    /(src|href|data-[a-z0-9-]+|poster|action)\s*=\s*(['"])(.*?)\2/gi,
    (match, attr, quote, val) => {
      if (shouldSkip(val)) return match;
      try {
        const abs = new URL(val.trim(), base).href;
        const local = urlToLocal.get(abs);

        if (local) return `${attr}=${quote}${local}${quote}`;

        // Preserve external links for href/action
        if (attr.toLowerCase() === 'href' || attr.toLowerCase() === 'action') {
          return `${attr}=${quote}${abs}${quote}`;
        }
        return match;
      } catch { return match; }
    }
  );

  // srcset cases
  html = html.replace(/(srcset|data-srcset|data-framer-srcset|data-framer-optimized-srcset)\s*=\s*(['"])(.*?)\2/gi, (match, attr, quote, srcset) => {
    const rewritten = srcset.split(',').map(entry => {
      const parts = entry.trim().split(/\s+/);
      const url = parts[0];
      if (!url || shouldSkip(url)) return entry;
      try {
        const abs = new URL(url, base).href;
        const local = urlToLocal.get(abs);
        return local ? [local, ...parts.slice(1)].join(' ') : entry;
      } catch { return entry; }
    }).join(', ');
    return `${attr}=${quote}${rewritten}${quote}`;
  });

  // url() in inline styles and SVG
  html = html.replace(/url\(\s*(['"]?)(.*?)\1\s*\)/gi, (match, q, val) => {
    if (shouldSkip(val)) return match;
    try {
      const abs = new URL(val.trim(), base).href;
      const local = urlToLocal.get(abs);
      return local ? `url("${local}")` : match;
    } catch { return match; }
  });

  // SVG specific image links
  html = html.replace(/(href|xlink:href)\s*=\s*(['"])(.*?)\2/gi, (match, attr, quote, val) => {
    if (shouldSkip(val)) return match;
    try {
      const abs = new URL(val.trim(), base).href;
      const local = urlToLocal.get(abs);
      return local ? `${attr}=${quote}${local}${quote}` : match;
    } catch { return match; }
  });

  // Remove <base> so local relative paths work
  html = html.replace(/<base\b[^>]*>/gi, '');
  return addCleanModeComment(html);
}

function addCleanModeComment(html) {
  const headMatch = html.match(/<head>/i);
  if (headMatch) {
    return html.replace(/<head>/i, `<head>\n    <!-- Extracted with ZipIt - Clean Mode -->`);
  }
  return html;
}

/** Lite mode — only rewrite <link href> and <script src> */
function rewriteHtmlLite(html, baseUrl, urlToLocal) {
  const base = new URL(baseUrl);
  html = html.replace(
    /(src|href)\s*=\s*(['"])(.*?)\2/gi,
    (match, attr, quote, val) => {
      if (shouldSkip(val)) return match;
      try {
        const abs = new URL(val.trim(), base).href;
        const local = urlToLocal.get(abs);

        if (local) return `${attr}=${quote}${local}${quote}`;

        // Ensure links to other pages point to the original site, not local 404s
        if (attr.toLowerCase() === 'href') {
          return `${attr}=${quote}${abs}${quote}`;
        }

        return match;
      } catch { return match; }
    }
  );
  html = html.replace(/<base\b[^>]*>/gi, '');
  return addCleanModeComment(html);
}

/** Rewrite url() references inside a CSS file */
function rewriteCss(css, cssUrl, urlToLocal) {
  const base = new URL(cssUrl);
  return css.replace(/url\(\s*(['"]?)(.*?)\1\s*\)/gi, (match, q, val) => {
    if (shouldSkip(val)) return match;
    try {
      const abs = new URL(val.trim(), base).href;
      const local = urlToLocal.get(abs);
      if (!local) return match;

      // CSS lives in css/ — make path relative from there
      let rel = "";
      if (local.startsWith('css/')) {
        rel = local.slice(4);
      } else {
        rel = '../' + local;
      }
      return `url("${rel}")`;
    } catch { return match; }
  });
}

function shouldSkip(val) {
  if (!val) return true;
  const v = val.trim();
  return !v || v.startsWith('data:') || v.startsWith('blob:') ||
    v.startsWith('#') || v.startsWith('javascript:') || v.startsWith('mailto:');
}

// ── Local path builder ────────────────────────────────────────
const usedPaths = new Set();
const FOLDER = { css: 'css/', js: 'js/', images: 'images/', fonts: 'fonts/', media: 'media/' };
const EXT = { css: '.css', js: '.js', images: '.png', fonts: '.woff2', media: '.mp4' };

function makeLocalPath(url, type) {
  const folder = FOLDER[type] || 'assets/';
  let filename = 'file' + (EXT[type] || '');

  try {
    const u = new URL(url);
    let pathname = decodeURIComponent(u.pathname);

    // 1. Strip query and hash from path
    pathname = pathname.split('?')[0].split('#')[0];

    // 2. Get last segment
    let segment = pathname.split('/').filter(Boolean).pop() || 'file';

    // 3. AGGRESSIVE HASH STRIPPING
    segment = segment
      .replace(/[.-][a-f0-9]{7,}(?=\.|$)/gi, '')
      .replace(/\.[a-f0-9]{7,}\./gi, '.');

    // 4. Ensure Extension
    let nameParts = segment.split('.');
    let currentExt = nameParts.length > 1 ? '.' + nameParts.pop().toLowerCase() : '';
    let baseName = nameParts.join('.') || segment;

    const validImage = ['.png', '.jpg', '.jpeg', '.svg', '.webp', '.gif', '.ico'];
    const validFont = ['.woff', '.woff2', '.ttf', '.otf', '.eot'];
    const validStyle = ['.css'];
    const validScript = ['.js'];

    let isCorrectExt = false;
    if (type === 'images' && validImage.includes(currentExt)) isCorrectExt = true;
    if (type === 'fonts' && validFont.includes(currentExt)) isCorrectExt = true;
    if (type === 'css' && validStyle.includes(currentExt)) isCorrectExt = true;
    if (type === 'js' && validScript.includes(currentExt)) isCorrectExt = true;

    if (!isCorrectExt) {
      if (type === 'fonts') currentExt = '.woff2';
      else if (type === 'images') currentExt = '.png';
      else if (type === 'css') currentExt = '.css';
      else if (type === 'js') currentExt = '.js';
      else currentExt = EXT[type] || '';
    }

    // 5. GIBBERISH DETECTION (Heuristic)
    const safeName = baseName || "file";
    const isGibberish = safeName.length > 15 && (!/[aeiou]/i.test(safeName) || (safeName.match(/\d/g) || []).length > safeName.length / 2);

    if (isGibberish || safeName.toLowerCase() === 'file') {
      baseName = `${type}-asset-${Math.random().toString(36).slice(2, 6)}`;
    }

    // 6. SLUGIFY
    filename = baseName.toLowerCase()
      .replace(/[^a-z0-9]/g, '-')
      .replace(/-+/g, '-')
      .replace(/^-|-$/g, '') + currentExt;

  } catch (e) {
    filename = `${type}-asset-${Math.random().toString(36).slice(2, 6)}${EXT[type] || ''}`;
  }

  if (filename.startsWith('.')) filename = "asset" + filename;

  // 7. DUPLICATE PROTECTION
  let path = folder + filename, n = 1;
  while (usedPaths.has(path)) {
    const dotIdx = filename.lastIndexOf('.');
    const stem = dotIdx > 0 ? filename.slice(0, dotIdx) : filename;
    const ext = dotIdx > 0 ? filename.slice(dotIdx) : '';
    path = `${folder}${stem}-${n}${ext}`;
    n++;
  }
  usedPaths.add(path);
  return path;
}

// ── Misc helpers ──────────────────────────────────────────────
function getSelectedTypes() {
  return ['css', 'js', 'images', 'fonts', 'media']
    .filter(t => document.getElementById(`toggle-${t}`)?.checked);
}

function fetchWithTimeout(url, ms) {
  const ctrl = new AbortController();
  const id = setTimeout(() => ctrl.abort(), ms);
  return fetch(url, { cache: 'no-store', signal: ctrl.signal })
    .finally(() => clearTimeout(id));
}

function setLoading(mode, on) {
  const btn = document.getElementById(`btn-${mode}`);
  const textEl = document.getElementById(`text-${mode}`);
  const iconEl = document.getElementById(`icon-${mode}`);
  const spId = `spinner-${mode}`;
  if (on) {
    btn.disabled = true;
    iconEl.style.display = 'none';
    if (!document.getElementById(spId)) {
      const sp = document.createElement('div');
      sp.className = 'spinner'; sp.id = spId;
      btn.insertBefore(sp, textEl);
    }
    textEl.textContent = 'Working…';
  } else {
    btn.disabled = false;
    document.getElementById(spId)?.remove();
    iconEl.style.display = '';
    textEl.textContent = mode === 'lite' ? 'Lite Download' : 'Full Offline Download';
  }
}

function showProgress(label) {
  document.getElementById('progress-wrap').classList.add('visible');
  document.getElementById('progress-label').textContent = label;
}
function hideProgress() { document.getElementById('progress-wrap').classList.remove('visible'); }
function updateProgress(pct) {
  document.getElementById('progress-fill').style.width = pct + '%';
  document.getElementById('progress-pct').textContent = Math.round(pct) + '%';
}
function showStatus(msg, type) {
  const el = document.getElementById('status');
  el.textContent = msg; el.className = `visible ${type}`;
}
function hideStatus() { document.getElementById('status').className = ''; }
function formatBytes(b) {
  if (b < 1024) return b + ' B';
  if (b < 1048576) return (b / 1024).toFixed(1) + ' KB';
  return (b / 1048576).toFixed(2) + ' MB';
}

function launchConfetti() {
  const count = 100;
  const defaults = { origin: { y: 0.7 } };

  function fire(particleRatio, opts) {
    const particles = Math.floor(count * particleRatio);
    for (let i = 0; i < particles; i++) {
      createParticle(opts);
    }
  }

  fire(0.25, { spread: 26, startVelocity: 55, });
  fire(0.2, { spread: 60, });
  fire(0.35, { spread: 100, decay: 0.91, scalar: 0.8 });
}

function createParticle(opts) {
  const p = document.createElement('div');
  p.style.position = 'fixed';
  p.style.zIndex = '10001';
  p.style.width = '10px';
  p.style.height = '10px';
  p.style.backgroundColor = ['#ff5c1a', '#3b82f6', '#22c55e', '#ffffff'][Math.floor(Math.random() * 4)];
  p.style.left = '50%';
  p.style.top = '100%';
  p.style.borderRadius = '2px';
  document.body.appendChild(p);

  const angle = (Math.random() * 60 + 240) * (Math.PI / 180);
  const velocity = Math.random() * 20 + 20;
  let vx = Math.cos(angle) * velocity;
  let vy = Math.sin(angle) * velocity;
  let x = window.innerWidth / 2;
  let y = window.innerHeight;

  function update() {
    x += vx;
    y += vy;
    vy += 0.8; // Gravity
    p.style.transform = `translate(${x - window.innerWidth / 2}px, ${y - window.innerHeight}px) rotate(${y}deg)`;
    if (y < window.innerHeight + 100) {
      requestAnimationFrame(update);
    } else {
      p.remove();
    }
  }
  requestAnimationFrame(update);
}

// ── DOM scanner — injected into the tab ──────────────────────
function extractAssets(selectedTypes) {
  const assets = [], seen = new Set(), base = document.baseURI;

  function add(url, type) {
    if (!url) return;
    const v = url.trim();
    if (!v || v.startsWith('data:') || v.startsWith('blob:') ||
      v.startsWith('#') || v.startsWith('javascript:')) return;
    try {
      const abs = new URL(v, base).href;
      if (!seen.has(abs)) { seen.add(abs); assets.push({ url: abs, type }); }
    } catch { }
  }

  if (selectedTypes.includes('css')) {
    document.querySelectorAll('link[rel="stylesheet"][href]').forEach(el => add(el.href, 'css'));
    document.querySelectorAll('link[rel="preload"][as="style"][href]').forEach(el => add(el.href, 'css'));
  }

  if (selectedTypes.includes('js')) {
    document.querySelectorAll('script[src]').forEach(el => add(el.src, 'js'));
    document.querySelectorAll('link[rel="modulepreload"][href]').forEach(el => add(el.href, 'js'));
    document.querySelectorAll('script:not([src])').forEach(el => {
      const matches = el.textContent.match(/import\s*\(["'](.+?)["']\)/g);
      if (matches) matches.forEach(m => add(m.match(/["'](.+?)["']/)[1], 'js'));
    });
  }

  if (selectedTypes.includes('images')) {
    document.querySelectorAll('img[src]').forEach(el => add(el.src, 'images'));
    document.querySelectorAll('img[srcset],source[srcset]').forEach(el =>
      el.srcset.split(',').forEach(s => add(s.trim().split(/\s+/)[0], 'images'))
    );
    document.querySelectorAll('picture source[src]').forEach(el => add(el.src, 'images'));
    document.querySelectorAll('video[poster]').forEach(el => add(el.poster, 'images'));
    document.querySelectorAll('image[href],image[xlink:href]').forEach(el => add(el.getAttribute('href') || el.getAttribute('xlink:href'), 'images'));

    document.querySelectorAll('[style]').forEach(el => {
      const style = el.getAttribute('style') || '';
      for (const m of style.matchAll(/url\(["']?(.+?)["']?\)/g)) add(m[1], 'images');
    });

    document.querySelectorAll('[data-src],[data-lazy],[data-bg],[data-framer-src],[data-framer-bg],[data-original]').forEach(el => {
      add(el.dataset.src || el.dataset.lazy || el.dataset.bg || el.dataset.framerSrc || el.dataset.framerBg || el.dataset.original, 'images');
    });

    // Aggressive Deep Scan: Every attribute of every element
    document.querySelectorAll('*').forEach(el => {
      for (const attr of el.attributes) {
        const val = attr.value;
        if (!val || val.length < 4) continue;
        if (val.startsWith('data:') || val.startsWith('blob:') || val.startsWith('#') || val.startsWith('javascript:')) continue;

        const isUrl = val.startsWith('http') || val.startsWith('/') || val.startsWith('./');
        if (!isUrl) continue;

        const isImg = /\.(png|jpg|jpeg|webp|gif|svg|ico)(\?|$)/i.test(val);
        const isFont = /\.(woff2?|ttf|otf|eot)(\?|$)/i.test(val);
        const isJS = /\.js(\?|$)/i.test(val);
        const isCSS = /\.css(\?|$)/i.test(val);

        if (isImg) add(val, 'images');
        else if (isFont) add(val, 'fonts');
        else if (isJS && selectedTypes.includes('js')) add(val, 'js');
        else if (isCSS && selectedTypes.includes('css')) add(val, 'css');
      }
    });
  }

  if (selectedTypes.includes('fonts')) {
    try {
      for (const sheet of document.styleSheets) {
        try {
          for (const rule of sheet.cssRules || []) {
            if (rule.type === CSSRule.FONT_FACE_RULE) {
              const src = rule.style.getPropertyValue('src');
              for (const m of src.matchAll(/url\(["']?(.+?)["']?\)/g)) add(m[1], 'fonts');
            }
          }
        } catch { }
      }
    } catch { }
  }

  if (selectedTypes.includes('images') || selectedTypes.includes('fonts')) {
    try {
      for (const sheet of document.styleSheets) {
        try {
          for (const rule of sheet.cssRules || []) {
            const text = rule.cssText || '';
            for (const m of text.matchAll(/url\(["']?([^)"']+)["']?\)/g)) {
              const u = m[1].trim();
              if (!u || u.startsWith('data:')) continue;
              const isFont = /\.(woff2?|ttf|otf|eot)(\?|$)/i.test(u);
              if (isFont && selectedTypes.includes('fonts')) add(u, 'fonts');
              if (!isFont && selectedTypes.includes('images')) add(u, 'images');
            }
          }
        } catch { }
      }
    } catch { }
  }

  if (selectedTypes.includes('media')) {
    document.querySelectorAll('video[src],audio[src],source[src],track[src]').forEach(el => add(el.src, 'media'));
    document.querySelectorAll('link[as="fetch"][href],link[as="json"][href]').forEach(el => add(el.href, 'media'));
  }

  // Performance API Deep Scan
  try {
    const resources = performance.getEntriesByType('resource');
    for (const res of resources) {
      const u = res.name;
      if (u.startsWith('data:') || u.includes('google-analytics') || u.includes('doubleclick')) continue;

      const isImg = /\.(png|jpg|jpeg|webp|gif|svg|ico)(\?|$)/i.test(u);
      const isFont = /\.(woff2?|ttf|otf|eot)(\?|$)/i.test(u);
      const isJS = /\.js(\?|$)/i.test(u);
      const isCSS = /\.css(\?|$)/i.test(u);

      if (isJS && selectedTypes.includes('js')) add(u, 'js');
      else if (isImg && selectedTypes.includes('images')) add(u, 'images');
      else if (isCSS && selectedTypes.includes('css')) add(u, 'css');
      else if (isFont && selectedTypes.includes('fonts')) add(u, 'fonts');
    }
  } catch (e) { }

  return assets;
}
